<?php

header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
header("Content-Type: application/json; charset=UTF-8");

include '../connection.php';

try {
    // Handle preflight request (CORS)
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(200);
        exit;
    }

    // Validate POST parameters
    if (empty($_POST['user_id']) || empty($_POST['order_id']) || empty($_POST['payment_method'])) {
        throw new Exception("Missing required parameters");
    }

    $user_id = intval($_POST['user_id']);
    $order_id = intval($_POST['order_id']);
    $payment_method = trim($_POST['payment_method']);

    if ($user_id <= 0 || $order_id <= 0) {
        throw new Exception("Invalid parameter values");
    }

    // Start transaction (ensure atomicity)
    $conn->begin_transaction();

    // 1️⃣ Delete cart items
    $deleteCart = $conn->prepare("DELETE FROM tblcart WHERE user_id = ?");
    if (!$deleteCart) throw new Exception("Prepare failed for DELETE tblcart: " . $conn->error);
    $deleteCart->bind_param("i", $user_id);
    $deleteCart->execute();
    if ($deleteCart->error) throw new Exception("Failed to delete cart: " . $deleteCart->error);

    // 2️⃣ Update order payment and status
    $payment_status = ($payment_method === 'cod') ? 'Paid by cod' : 'Paid';
    $updateOrder = $conn->prepare("UPDATE tbl_order SET status = 'Submitted', payment_status = ? WHERE id = ? AND user_id = ?");
    if (!$updateOrder) throw new Exception("Prepare failed for UPDATE tbl_order: " . $conn->error);
    $updateOrder->bind_param("sii", $payment_status, $order_id, $user_id);
    $updateOrder->execute();
    if ($updateOrder->affected_rows === 0) {
        throw new Exception("No matching order found or already updated");
    }

    // ✅ Commit transaction
    $conn->commit();

    echo json_encode([
        "success" => true,
        "message" => "Payment processed successfully and cart cleared."
    ]);

} catch (Exception $e) {
    // Rollback transaction on any error
    if ($conn->in_transaction) {
        $conn->rollback();
    }

    echo json_encode([
        "success" => false,
        "message" => $e->getMessage()
    ]);
    exit;
}


/*
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");

include '../connection.php';

try {
    if (!isset($_POST['user_id'], $_POST['order_id'], $_POST['payment_method'])) {
        throw new Exception("Missing parameters");
    }

    $user_id = intval($_POST['user_id']);
    $order_id = intval($_POST['order_id']);
    $payment_method = trim($_POST['payment_method']);

    if ($user_id <= 0 || $order_id <= 0 || empty($payment_method)) {
        throw new Exception("Invalid parameter values");
    }

    // Start transaction safely
    $conn->begin_transaction();

    // Delete cart items
    $deleteCart = $conn->prepare("DELETE FROM tblcart WHERE user_id = ?");
    if (!$deleteCart) throw new Exception("Prepare failed for DELETE tblcart: " . $conn->error);
    $deleteCart->bind_param("i", $user_id);
    $deleteCart->execute();

    if ($deleteCart->error) throw new Exception("Failed to delete cart: " . $deleteCart->error);

    // Update order
    $payment_status = ($payment_method === 'cod') ? 'Paid by cod' : 'Paid';
    $updateOrder = $conn->prepare("UPDATE tbl_order SET status='Submitted', payment_status=? WHERE id=? AND user_id=?");
    if (!$updateOrder) throw new Exception("Prepare failed for UPDATE tbl_order: " . $conn->error);
    $updateOrder->bind_param("sii", $payment_status, $order_id, $user_id);
    $updateOrder->execute();

    if ($updateOrder->error) throw new Exception("Failed to update order: " . $updateOrder->error);

    // Commit
    $conn->commit();

    echo json_encode([
        "success" => true,
        "message" => "Payment processed and cart cleared successfully"
    ]);

} catch (Exception $e) {
    if ($conn->in_transaction) $conn->rollback();

    echo json_encode([
        "success" => false,
        "message" => $e->getMessage()
    ]);
}
*/
?>
